<?php
header('Content-Type: application/json');
include('../config.php');
session_start();

// Ensure that the request method is POST
if ($_SERVER['REQUEST_METHOD'] != 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method.']);
    exit;
}

// CSRF check
if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    exit(json_encode(["status" => "error", "message" => "CSRF validation failed"]));
}

$response = ['success' => false, 'message' => 'Unknown error'];

// Simple role check (optional, can restrict deletes/edits if needed)
$role = $_SESSION['role'] ?? '';

// Get POST parameters
$action = $_POST['action'] ?? '';
$id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
$content_title = trim($_POST['content_title'] ?? '');
$content = $_POST['content'] ?? '';

// Sanitize content_title to remove HTML and scripts
$content_title = htmlspecialchars(strip_tags($content_title), ENT_QUOTES, 'UTF-8');

// Validate that required fields are not empty
if ($action!='delete' && (empty($content_title) || empty(trim(strip_tags($content))))) {
    $response['message'] = 'Content title and content cannot be empty.';
    echo json_encode($response);
    exit;
}

// Handle the delete action via POST (AJAX)
if ($action === 'delete' && $id > 0) {
    try {
        // Prepare and execute the soft delete query
        $stmt = $conn->prepare("UPDATE content SET deleted_at = NOW() WHERE content_id = ?");
        $stmt->bind_param("i", $id);
        
        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Content deleted successfully.'];
        } else {
            $response['message'] = 'Failed to delete content.';
        }

        $stmt->close();
    } catch (Exception $e) {
        $response['message'] = 'Error: ' . $e->getMessage();
    }

    // Return JSON response after the delete action
    echo json_encode($response);
    $conn->close();
    exit;
}

// For Add and Edit actions
try {
    if ($action === 'Add') {
        // Insert new content
        $stmt = $conn->prepare("INSERT INTO content (content_title, content) VALUES (?, ?)");
        $stmt->bind_param("ss", $content_title, $content);

        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Content added successfully', 'redirect' => 'contents.php'];
        } else {
            $response['message'] = 'Failed to add content';
        }

        $stmt->close();

    } elseif ($action === 'Edit' && $id > 0) {
        // Edit existing content
        $stmt = $conn->prepare("UPDATE content SET content_title=?, content=?, updated_at=NOW() WHERE content_id=?");
        $stmt->bind_param("ssi", $content_title, $content, $id);

        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Content updated successfully', 'redirect' => 'contents.php'];
        } else {
            $response['message'] = 'Failed to update content';
        }

        $stmt->close();

    } else {
        $response['message'] = 'Invalid action or missing content ID';
    }

} catch (Exception $e) {
    $response['message'] = 'Error: ' . $e->getMessage();
}

// Return JSON response
echo json_encode($response);
$conn->close();
exit;
?>


