<?php
header('Content-Type: application/json');
include('../config.php');
session_start();

// Ensure POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method.']);
    exit;
}

// CSRF check
if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    echo json_encode(['success' => false, 'message' => 'CSRF validation failed.']);
    exit;
}

$response = ['success' => false, 'message' => 'Unknown error.'];

// Read form data
$action = $_POST['action'] ?? '';
$id = isset($_POST['id']) ? (int)$_POST['id'] : 0;

$name = trim($_POST['name'] ?? '');
$description = trim($_POST['description'] ?? '');
$fees = trim($_POST['fees'] ?? '0');
$discount = trim($_POST['discount'] ?? '0');

// Validation
if (!in_array($action, ['delete', 'Activate', 'Deactivate']) && (empty($name) || empty($description))) {
    echo json_encode(['success' => false, 'message' => 'Course name and description are required.']);
    exit;
}


// Directory for uploads
$uploadDir = './assets/img/courses/';
if (!is_dir($uploadDir)) {
    mkdir($uploadDir, 0777, true);
}

$imagePath = null;

// Handle file upload (for Add and Edit)
if (!empty($_FILES['image']['name'])) {
    $fileName = time() . '_' . preg_replace('/[^A-Za-z0-9._-]/', '_', $_FILES['image']['name']);
    $targetFile = $uploadDir . $fileName;
    $fileType = strtolower(pathinfo($targetFile, PATHINFO_EXTENSION));

    // Validate image type
    $allowedTypes = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
    if (!in_array($fileType, $allowedTypes)) {
        echo json_encode(['success' => false, 'message' => 'Invalid image format. Only JPG, PNG, GIF, WEBP allowed.']);
        exit;
    }

    // Move file
    if (move_uploaded_file($_FILES['image']['tmp_name'], $targetFile)) {
        $imagePath = 'assets/img/courses/' . $fileName;

        // If editing and there was an old image, remove it
        if (!empty($_POST['old_image']) && file_exists('../' . $_POST['old_image'])) {
            @unlink('../' . $_POST['old_image']);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to upload image.']);
        exit;
    }
}


// === DELETE COURSE ===
if ($action === 'delete' && $id > 0) {
    try {
        $stmt = $conn->prepare("UPDATE course SET deleted_at = NOW() WHERE id = ?");
        $stmt->bind_param("i", $id);
        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Course deleted successfully.'];
        } else {
            $response['message'] = 'Failed to delete course.';
        }
        $stmt->close();
    } catch (Exception $e) {
        $response['message'] = 'Error: ' . $e->getMessage();
    }

    echo json_encode($response);
    $conn->close();
    exit;
}

// === ACTIVATE / DEACTIVATE COURSE ===
if (($action === 'Activate' || $action === 'Deactivate') && $id > 0) {
    $newStatus = ($action === 'Activate') ? 'Active' : 'Deactive';
    try {
        $stmt = $conn->prepare("UPDATE course SET status = ?, updated_at = NOW() WHERE id = ?");
        $stmt->bind_param("si", $newStatus, $id);
        if ($stmt->execute()) {
            $response = [
                'success' => true,
                'message' => "Course {$newStatus}d successfully."
            ];
        } else {
            $response['message'] = "Failed to {$action} course.";
        }
        $stmt->close();
    } catch (Exception $e) {
        $response['message'] = 'Error: ' . $e->getMessage();
    }

    echo json_encode($response);
    $conn->close();
    exit;
}


// === ADD NEW COURSE ===
if ($action === 'Add') {
    try {
        $stmt = $conn->prepare("INSERT INTO course (name, description, fees, discount, image) VALUES (?, ?, ?, ?, ?)");
        $stmt->bind_param("ssdds", $name, $description, $fees, $discount, $imagePath);
        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Course added successfully.', 'redirect' => 'courses.php'];
        } else {
            $response['message'] = 'Failed to add course.';
        }
        $stmt->close();
    } catch (Exception $e) {
        $response['message'] = 'Error: ' . $e->getMessage();
    }
}

// === EDIT EXISTING COURSE ===
elseif ($action === 'Edit' && $id > 0) {
    try {
        if ($imagePath) {
            $stmt = $conn->prepare("UPDATE course SET name=?, description=?, fees=?, discount=?, image=?, updated_at=NOW() WHERE id=?");
            $stmt->bind_param("ssddsi", $name, $description, $fees, $discount, $imagePath, $id);
        } else {
            $stmt = $conn->prepare("UPDATE course SET name=?, description=?, fees=?, discount=?, updated_at=NOW() WHERE id=?");
            $stmt->bind_param("ssddi", $name, $description, $fees, $discount, $id);
        }

        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Course updated successfully.', 'redirect' => 'courses.php'];
        } else {
            $response['message'] = 'Failed to update course.';
        }

        $stmt->close();
    } catch (Exception $e) {
        $response['message'] = 'Error: ' . $e->getMessage();
    }
} else {
    $response['message'] = 'Invalid action or missing ID.';
}

echo json_encode($response);
$conn->close();
exit;
?>



