<?php
header('Content-Type: application/json');
include('../config.php');
session_start();

// Ensure POST method
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

// CSRF validation
if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    echo json_encode(['success' => false, 'message' => 'CSRF validation failed']);
    exit;
}

$response = ['success' => false, 'message' => 'Unknown error'];

// Collect input
$action        = $_POST['action'] ?? '';
$enquiry_id    = isset($_POST['id']) ? (int)$_POST['id'] : 0;
$name          = trim($_POST['name'] ?? '');
$email         = trim($_POST['email'] ?? '');
$phone         = trim($_POST['phone'] ?? '');
$reference     = trim($_POST['reference'] ?? '');
$subject       = trim($_POST['subject'] ?? '');
$message       = trim($_POST['message'] ?? '');
$source        = trim($_POST['source'] ?? 'Offline');
$assigned_to   = trim($_POST['assigned_to'] ?? '');
$status        = trim($_POST['status'] ?? 'New Enquiry');
$course_id     = trim($_POST['course_id'] ?? '');
$subject_desc  = trim($_POST['subject_description'] ?? '');

// Clean up inputs
function clean($val) {
    return htmlspecialchars(strip_tags($val), ENT_QUOTES, 'UTF-8');
}

$name         = clean($name);
$email        = clean($email);
$phone        = clean($phone);
$reference    = clean($reference);
$subject      = clean($subject);
$message      = clean($message);
$source       = clean($source);
$assigned_to  = clean($assigned_to);
$status       = clean($status);
$course_id    = clean($course_id);
$subject_desc = clean($subject_desc);

// --- Action Handling ---
try {
    if ($action === 'Add') {

        $stmt = $conn->prepare("INSERT INTO enquiries 
            (name, email, phone, reference, subject, course, subject_description, message, source, assigned_to, status, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");
        $stmt->bind_param("sssssssssss", $name, $email, $phone, $reference, $subject, $course_id, $subject_desc, $message, $source, $assigned_to, $status);

        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Enquiry added successfully', 'redirect' => 'enquiries.php'];
        } else {
            $response['message'] = 'Failed to add enquiry.';
        }
        $stmt->close();

    } elseif ($action === 'Edit' && $enquiry_id > 0) {

        $stmt = $conn->prepare("UPDATE enquiries 
            SET name=?, email=?, phone=?, reference=?, subject=?, course=?, subject_description=?, message=?, source=?, assigned_to=?, status=?, updated_at=NOW() 
            WHERE enquiry_id=?");
        $stmt->bind_param("sssssssssssi", $name, $email, $phone, $reference, $subject, $course_id, $subject_desc, $message, $source, $assigned_to, $status, $enquiry_id);

        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Enquiry updated successfully', 'redirect' => 'enquiries.php'];
        } else {
            $response['message'] = 'Failed to update enquiry.';
        }
        $stmt->close();

    } elseif ($action === 'delete' && $enquiry_id > 0) {

        $stmt = $conn->prepare("UPDATE enquiries SET deleted_at=NOW() WHERE enquiry_id=?");
        $stmt->bind_param("i", $enquiry_id);

        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Enquiry deleted successfully.'];
        } else {
            $response['message'] = 'Failed to delete enquiry.';
        }
        $stmt->close();

    } elseif ($action === 'assign' && $enquiry_id > 0) {
        if (($_SESSION['role'] ?? '') !== 'Admin') {
            $response = ['success' => false, 'message' => 'Unauthorized action. Only Admins can assign enquiries.'];
        } else {
            $stmt = $conn->prepare("UPDATE enquiries SET assigned_to=? WHERE enquiry_id=?");
            $stmt->bind_param("si", $assigned_to, $enquiry_id);

            if ($stmt->execute()) {
                $response = ['success' => true, 'message' => 'Enquiry assigned successfully.'];
            } else {
                $response['message'] = 'Failed to assign enquiry.';
            }
            $stmt->close();
        }

    } else {
        $response['message'] = 'Invalid request or action.';
    }
} catch (Exception $e) {
    $response['message'] = 'Error: ' . $e->getMessage();
}

echo json_encode($response);
$conn->close();
exit;
?>
