document.addEventListener('DOMContentLoaded', function () {
  const form = document.getElementById('galleryform');
  const typeSelect = document.getElementById('type');
  const imageDiv = document.getElementById('imageUploadDiv');
  const videoDiv = document.getElementById('videoLinkDiv');
  const videoInput = document.getElementById('video_link');
  const imageInput = document.getElementById('image');

  let previousType = typeSelect.value; // store initial type

  function toggleFields() {
    const val = typeSelect.value;
    if (val === 'Image') {
      imageDiv.style.display = 'block';
      videoDiv.style.display = 'none';
      videoInput.value = '';
      videoInput.removeAttribute('required');
    } else if (val === 'Video') {
      imageDiv.style.display = 'none';
      videoDiv.style.display = 'block';
      videoInput.setAttribute('required', 'required');
    } else {
      imageDiv.style.display = 'none';
      videoDiv.style.display = 'none';
      videoInput.removeAttribute('required');
    }
  }

  typeSelect.addEventListener('change', function () {
    toggleFields();
  });

  toggleFields();

  if (form) {
    form.addEventListener('submit', function (e) {
      e.preventDefault();

      const msgDiv = document.getElementById('msg');
      msgDiv.innerHTML = '';

      // ✅ Prevent submission if switching from Video → Image without image upload
      const currentType = typeSelect.value;
      const oldType = form.dataset.oldtype; // stored below from PHP
      if (
        oldType === 'Video' &&
        currentType === 'Image' &&
        (!imageInput.files || imageInput.files.length === 0)
      ) {
        msgDiv.innerHTML =
          '<div class="alert alert-danger">Please upload an image since you changed the type from Video to Image.</div>';
        return;
      }

      const submitBtn = form.querySelector('button[type="submit"]');
      submitBtn.disabled = true;
      submitBtn.textContent = 'Please wait...';

      const formData = new FormData(form);

      fetch('gallery-action.php', { method: 'POST', body: formData })
        .then((response) => response.json())
        .then((data) => {
          console.log('Response:', data);
          msgDiv.innerHTML = '';

          if (data.success) {
            msgDiv.innerHTML = `<div class="alert alert-success">${data.message}</div>`;
            if (data.redirect) {
              setTimeout(() => (window.location.href = data.redirect), 1500);
            }
          } else {
            msgDiv.innerHTML = `<div class="alert alert-danger">${data.message}</div>`;
          }
        })
        .catch((error) => {
          console.error('Error:', error);
          msgDiv.innerHTML =
            '<div class="alert alert-danger">Something went wrong. Please try again.</div>';
        })
        .finally(() => {
          submitBtn.disabled = false;
          submitBtn.textContent =
            form.querySelector('input[name="action"]').value === 'Edit'
              ? 'Update'
              : 'Submit';
        });
    });
  }
});


// --- Delete Content ---
function deleteGallery(galleryId) {
  if (!confirm('Are you sure you want to delete this Image/Video?')) {
    return;
  }

  var csrfToken = document.querySelector('input[name="csrf_token"]').value;

  var data = {
    action: 'delete',
    id: galleryId,
    csrf_token: csrfToken
  };

  var xhr = new XMLHttpRequest();
  xhr.open('POST', 'gallery-action.php', true);
  xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');

  xhr.onload = function () {
    if (xhr.status === 200) {
      var response = JSON.parse(xhr.responseText);
      if (response.success === true) {
        location.reload();
      } else {
        alert(response.message);
      }
    } else {
      alert('An error occurred while deleting the image/video.');
    }
  };

  xhr.send(
    'action=' +
      data.action +
      '&id=' +
      data.id +
      '&csrf_token=' +
      encodeURIComponent(data.csrf_token)
  );
}








