document.addEventListener("DOMContentLoaded", function() {

  /* =========================================================
     🔹 Generic AJAX form submission function
  ========================================================== */
  function ajaxSubmit(form, editors = []) {
    // Copy Quill editor contents (if any)
    editors.forEach(function(ed){
      const input = document.getElementById(ed.hiddenInputId);
      if(input) input.value = ed.quill.root.innerHTML;
    });

    // Validate if required
    if(!validateForm(form)) return;

    // Disable submit buttons
    const submitButtons = form.querySelectorAll('button[type="submit"]');
    submitButtons.forEach(btn => {
      btn.disabled = true;
      btn.dataset.originalText = btn.innerHTML;
      btn.innerHTML = 'Please wait...';
    });

    // Submit via AJAX
    const formData = new FormData(form);
    fetch('settings-action.php', { method: 'POST', body: formData })
      .then(res => res.json())
      .then(data => {
        // Re-enable buttons
        submitButtons.forEach(btn => {
          btn.disabled = false;
          btn.innerHTML = btn.dataset.originalText;
        });

        if(data.status === 'success'){
          alert(data.message);
          location.reload();
        } else {
          alert('Error: ' + data.message);
        }
      })
      .catch(err => {
        console.error(err);
        // Re-enable buttons
        submitButtons.forEach(btn => {
          btn.disabled = false;
          btn.innerHTML = btn.dataset.originalText;
        });
        alert('AJAX error');
      });
  }

  /* =========================================================
     🔹 Common validation for all forms
  ========================================================== */
  function validateForm(form) {
    const email = form.querySelector('input[name="email"]');
    const senderEmail = form.querySelector('input[name="sender_email"]');
    const phone = form.querySelector('input[name="phone"]');
    const whatsapp = form.querySelector('input[name="whatsapp"]');
    const regPrefix = form.querySelector('input[name="registration_no_prefix"]');
    let valid = true;

    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    const phoneRegex = /^[0-9]{10,15}$/;
    const prefixRegex = /^[A-Za-z0-9]{1,4}$/; // ✅ 1 to 4 alphanumeric chars only


    if(email && email.value.trim() !== '' && !emailRegex.test(email.value.trim())) {
      alert('Invalid Email');
      valid = false;
    }
    if(senderEmail && senderEmail.value.trim() !== '' && !emailRegex.test(senderEmail.value.trim())) {
      alert('Invalid Sender Email');
      valid = false;
    }
    if(phone && phone.value.trim() !== '' && !phoneRegex.test(phone.value.trim())) {
      alert('Invalid Phone Number');
      valid = false;
    }
    if(whatsapp && whatsapp.value.trim() !== '' && !phoneRegex.test(whatsapp.value.trim())) {
      alert('Invalid WhatsApp Number');
      valid = false;
    }
    
     // ✅ Validate registration_no_prefix
    if (regPrefix && regPrefix.value.trim() !== '' && !prefixRegex.test(regPrefix.value.trim())) {
      alert('Registration No Prefix must be 1-4 alphanumeric characters only (A-Z, 0-9).');
      valid = false;
    }
    
    
    return valid;
  }

  /* =========================================================
     🔹 Quill editor initialization helper
  ========================================================== */
  function initQuill(editorId, hiddenInputId, placeholderText) {
    const el = document.getElementById(editorId);
    if (!el) return null;
    const quill = new Quill('#' + editorId, { theme: 'snow', placeholder: placeholderText });
    return { quill, hiddenInputId };
  }

  /* =========================================================
     1️⃣ SETTINGS: GENERAL PAGE
  ========================================================== */
  const formGeneral = document.getElementById('settings_general');
  if (formGeneral) {
    const editorsGeneral = [
      initQuill('privacy_policy_editor', 'privacy_policy', 'Write your Privacy Policy here...'),
      initQuill('term_editor', 'term_and_conditions', 'Write Terms & Conditions here...'),
      initQuill('disclaimer_editor', 'disclaimer', 'Write Disclaimer here...')
    ].filter(e => e !== null);

    formGeneral.addEventListener('submit', function(e){
      e.preventDefault();
      ajaxSubmit(formGeneral, editorsGeneral);
    });
  }

  /* =========================================================
     2️⃣ SETTINGS: WEBSITE PAGE
  ========================================================== */
  const formWebsite = document.getElementById('settings_website');
  if (formWebsite) {
    formWebsite.addEventListener('submit', function(e){
      e.preventDefault();
      ajaxSubmit(formWebsite);
    });
  }

  /* =========================================================
     3️⃣ SETTINGS: SEO PAGE
  ========================================================== */
  const formSEO = document.getElementById('settings_seo');
  if (formSEO) {
    const editorsSEO = [
      initQuill('footer_editor', 'footer_scripts', 'Paste Footer Scripts here...'),
      initQuill('header_editor', 'head_scripts', 'Paste Header Scripts here...')
    ].filter(e => e !== null);

    formSEO.addEventListener('submit', function(e){
      e.preventDefault();
      ajaxSubmit(formSEO, editorsSEO);
    });
  }

  /* =========================================================
     4️⃣ SETTINGS: EMAIL CONFIGURATION PAGE
  ========================================================== */
  const formEmails = document.getElementById('settings_emails');
  if (formEmails) {
    formEmails.addEventListener('submit', function(e){
      e.preventDefault();
      ajaxSubmit(formEmails);
    });
  }

  /* =========================================================
     5️⃣ SETTINGS: PAYMENT CONFIGURATION PAGE
  ========================================================== */
  const formPayment = document.getElementById('settings_payment');
  if (formPayment) {
    formPayment.addEventListener('submit', function(e){
      e.preventDefault();
      ajaxSubmit(formPayment);
    });
  }

  /* =========================================================
     6️⃣ SETTINGS: WHATSAPP CONFIGURATION PAGE
  ========================================================== */
  const formWhatsapp = document.getElementById('settings_whatsapp');
  if (formWhatsapp) {
    formWhatsapp.addEventListener('submit', function(e){
      e.preventDefault();
      ajaxSubmit(formWhatsapp);
    });
  }

});





