<?php
header('Content-Type: application/json');
include('../config.php');
session_start();

$response = ['success' => false, 'message' => 'Unknown error'];

// Ensure POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

// CSRF check
if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    exit(json_encode(["success" => false, "message" => "CSRF validation failed"]));
}

// Helper function
function sanitize($str) {
    return htmlspecialchars(trim($str), ENT_QUOTES, 'UTF-8');
}

$action = sanitize($_POST['action'] ?? '');
$id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
$name = sanitize($_POST['name'] ?? '');
$email = sanitize($_POST['email'] ?? '');
$phone = sanitize($_POST['mobile'] ?? '');
$reference = sanitize($_POST['reference'] ?? '');
$dob = sanitize($_POST['dob'] ?? '');
$password = '123456';

// Validation
if ($action != 'delete' && (empty($name) || empty($email) || empty($phone) || empty($dob))) {
    echo json_encode(['success' => false, 'message' => 'Please fill all required fields.']);
    exit;
}
if ($action != 'delete' && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
    echo json_encode(['success' => false, 'message' => 'Invalid email format.']);
    exit;
}
if ($action != 'delete' && !preg_match('/^[0-9]{10}$/', $phone)) {
    echo json_encode(['success' => false, 'message' => 'Enter valid 10 digit mobile number.']);
    exit;
}

// Check mobile uniqueness
$stmt = $conn->prepare("SELECT id FROM student WHERE mobile = ?" . ($action === 'Edit' ? " AND id != ?" : ""));
if ($action === 'Edit') {
    $stmt->bind_param("si", $phone, $id);
} else {
    $stmt->bind_param("s", $phone);
}
$stmt->execute();
$stmt->store_result();
if ($stmt->num_rows > 0) {
    $stmt->close();
    echo json_encode(['success' => false, 'message' => 'Mobile already exists for another student.']);
    exit;
}
$stmt->close();

// Check email uniqueness
$stmt = $conn->prepare("SELECT id FROM student WHERE email = ?" . ($action === 'Edit' ? " AND id != ?" : ""));
if ($action === 'Edit') {
    $stmt->bind_param("si", $email, $id);
} else {
    $stmt->bind_param("s", $email);
}
$stmt->execute();
$stmt->store_result();

if ($stmt->num_rows > 0) {
    $stmt->close();
    echo json_encode(['success' => false, 'message' => 'Email already exists for another student.']);
    exit;
}
$stmt->close();


try {
    // Upload helper
    function uploadFile($fieldName, $oldFile = '', $subdir = 'students') {
        if (!empty($_FILES[$fieldName]['name'])) {
            $file_name = time() . '_' . basename($_FILES[$fieldName]['name']);
            $target_dir = "../uploads/$subdir/";
            if (!file_exists($target_dir)) mkdir($target_dir, 0777, true);
            $target_path = $target_dir . $file_name;

            if (move_uploaded_file($_FILES[$fieldName]['tmp_name'], $target_path)) {
                return $target_path;
            }
        }
        return $oldFile;
    }

    // Handle ADD
    if ($action === 'Add') {

        // Generate registration number
        $prefix = $settings['registration_no_prefix'] ?? 'REG';
        $month = date('m');
        $year = date('y');
        $last4 = substr($phone, -4);
        $registration_no = $prefix . $month . $year . $last4;

        // Handle photo upload (optional)
        $photo_path = '';
        if (!empty($_FILES['photo']['name'])) {
            $photo_path = uploadFile('photo');
        }

        // Handle student_form (optional)
        $student_form_path = '';
        if (!empty($_FILES['student_form']['name'])) {
            $student_form_path = uploadFile('student_form');
        }

        // Handle student_doc (optional)
        $student_doc_path = '';
        if (!empty($_FILES['student_doc']['name'])) {
            $student_doc_path = uploadFile('student_doc');
        }

        // Insert
        $stmt = $conn->prepare("INSERT INTO student (registration_no, name, email, mobile, dob, photo, student_form, student_doc, reference, password, status, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'Active', NOW(), NOW())");
        $stmt->bind_param("ssssssssss", $registration_no, $name, $email, $phone, $dob, $photo_path, $student_form_path, $student_doc_path, $reference, $password);

        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Student added successfully', 'redirect' => 'students.php'];
        } else {
            $response['message'] = 'Failed to add student: ' . $stmt->error;
        }
        $stmt->close();
    }

    // Handle EDIT
    elseif ($action === 'Edit' && $id > 0) {

        $photo_path = uploadFile('photo', $_POST['old_photo'] ?? '');
        $student_form_path = uploadFile('student_form', $_POST['old_student_form'] ?? '');
        $student_doc_path = uploadFile('student_doc', $_POST['old_student_doc'] ?? '');

        $stmt = $conn->prepare("UPDATE student SET name=?, email=?, mobile=?, dob=?, photo=?, student_form=?, student_doc=?, reference=?, updated_at=NOW() WHERE id=?");
        $stmt->bind_param("ssssssssi", $name, $email, $phone, $dob, $photo_path, $student_form_path, $student_doc_path, $reference, $id);

        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Student updated successfully', 'redirect' => 'students.php'];
        } else {
            $response['message'] = 'Failed to update student: ' . $stmt->error;
        }
        $stmt->close();
    }

    // Handle DELETE
    elseif ($action === 'delete' && $id > 0) {
        $stmt = $conn->prepare("DELETE FROM student WHERE id = ?");
        $stmt->bind_param("i", $id);

        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Student deleted successfully.'];
        } else {
            $response['message'] = 'Failed to delete student.';
        }

        $stmt->close();
    } else {
        $response['message'] = 'Invalid action or missing student ID';
    }

} catch (Exception $e) {
    $response['message'] = 'Error: ' . $e->getMessage();
}

echo json_encode($response);
$conn->close();
exit;
?>
