<?php
include("auth.php");
include("../config.php");
header('Content-Type: application/json');

// ✅ CSRF Validation
if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
    echo json_encode(['success' => false, 'message' => 'Invalid CSRF token.']);
    exit;
}

$action = $_POST['action'] ?? '';
$response = ['success' => false, 'message' => 'Invalid request.'];

// ✅ Helper
function clean($data) {
    return htmlspecialchars(trim($data), ENT_QUOTES, 'UTF-8');
}

// ✅ Upload directory (physical path)
$uploadDir = './assets/student-corner/';
if (!file_exists($uploadDir)) {
    mkdir($uploadDir, 0777, true);
}

// ------------------ ADD / EDIT ------------------
if ($action === 'Add' || $action === 'Edit') {
    $id = $_POST['id'] ?? null;
    $title = clean($_POST['title'] ?? '');
    $type = clean($_POST['type'] ?? '');
    $restriction = clean($_POST['restriction'] ?? '');
    $link = null;

    if (empty($title) || empty($type)) {
        echo json_encode(['success' => false, 'message' => 'Please fill in all required fields.']);
        exit;
    }


    // 🧠 Handle link or file based on type
if (in_array($type, ['Videos', 'Links', 'Quiz', 'Live'])) {
    $link = clean($_POST['link'] ?? '');
    if (empty($link)) {
        echo json_encode(['success' => false, 'message' => 'Please enter a valid link.']);
        exit;
    }
} else {
    // 🧾 File-based upload
    $fileUploaded = isset($_FILES['fileUpload']) && $_FILES['fileUpload']['error'] === UPLOAD_ERR_OK;

    if ($fileUploaded) {
        $filename = time() . '_' . basename($_FILES['fileUpload']['name']);
        $targetFile = $uploadDir . $filename;

        if (move_uploaded_file($_FILES['fileUpload']['tmp_name'], $targetFile)) {
            // Save path in DB without "./"
            $link = 'assets/student-corner/' . $filename;
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to upload file.']);
            exit;
        }
    } else {
        if ($action === 'Edit') {
            // Get existing record
            $stmt = $conn->prepare("SELECT type, link FROM student_corner WHERE id = ?");
            $stmt->bind_param("i", $id);
            $stmt->execute();
            $res = $stmt->get_result()->fetch_assoc();
            $stmt->close();

            // 🧠 CASE 1: Editing same file-type content
            if (!in_array($res['type'], ['Videos', 'Links','Quiz','Live']) && !empty($res['link'])) {
                $link = $res['link']; // Keep existing file
            }
            // 🧠 CASE 2: Changing from link/video → file-based (require new file)
            else {
                echo json_encode(['success' => false, 'message' => 'Please upload a file for this type.']);
                exit;
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Please upload a file.']);
            exit;
        }
    }
}


    // 🧩 Insert or update
    if ($action === 'Add') {
        $stmt = $conn->prepare("INSERT INTO student_corner (title, type, link,restriction) VALUES (?, ?, ?, ?)");
        $stmt->bind_param("ssss", $title, $type, $link, $restriction);
        $success = $stmt->execute();
        $stmt->close();
    } else {
        $stmt = $conn->prepare("UPDATE student_corner SET title=?, type=?, link=?, restriction=?, updated_at=NOW() WHERE id=?");
        $stmt->bind_param("ssssi", $title, $type, $link, $restriction, $id);
        $success = $stmt->execute();
        $stmt->close();
    }

    if ($success) {
        $response = [
            'success' => true,
            'message' => ($action === 'Add' ? 'File/Link added successfully!' : 'File/Link updated successfully!'),
            'redirect' => 'student-corner-library.php'
        ];
    } else {
        $response['message'] = 'Database operation failed. Please try again.';
    }
}

// ------------------ DELETE ------------------
elseif ($action === 'delete') {
    $id = intval($_POST['id']);
    $stmt = $conn->prepare("UPDATE student_corner SET deleted_at = NOW() WHERE id = ?");
    $stmt->bind_param("i", $id);
    $response['success'] = $stmt->execute();
    $response['message'] = $response['success'] ? 'File/Link deleted successfully!' : 'Failed to delete file/link.';
    $stmt->close();
}

// ------------------ ACTIVATE / DEACTIVATE ------------------
elseif ($action === 'Activate' || $action === 'Deactivate') {
    $id = intval($_POST['id']);
    $newStatus = $action === 'Activate' ? 'Active' : 'Inactive';
    $stmt = $conn->prepare("UPDATE student_corner SET status = ?, updated_at = NOW() WHERE id = ?");
    $stmt->bind_param("si", $newStatus, $id);
    $response['success'] = $stmt->execute();
    $response['message'] = $response['success']
        ? "File/Link {$newStatus}d successfully!"
        : "Failed to update status.";
    $stmt->close();
}

echo json_encode($response);
?>
