<?php
include("../config.php");
session_start();
header('Content-Type: application/json');

// Security checks
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

$id = intval($_POST['id'] ?? 0);
$status = $_POST['status'] ?? '';
$csrf = $_POST['csrf_token'] ?? '';

if (!$id || !in_array($status, ['Paid', 'Failed'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid request data.']);
    exit;
}

// CSRF protection
if ($csrf !== ($_SESSION['csrf_token'] ?? '')) {
    echo json_encode(['success' => false, 'message' => 'CSRF mismatch.']);
    exit;
}

try {
    // === Step 1: Update payment_status in fees_receipts ===
    $stmt = $conn->prepare("UPDATE fees_receipts SET payment_status = ?, updated_at = NOW() WHERE id = ?");
    $stmt->bind_param("si", $status, $id);
    if (!$stmt->execute()) {
        throw new Exception("Failed to update receipt status.");
    }
    $stmt->close();

    // === Step 2: Fetch enrolment_id and batch id from this receipt ===
    $enrolment_id = $bid = 0;
    $stmtInfo = $conn->prepare("SELECT enrolment_id, bid FROM fees_receipts WHERE id = ?");
    $stmtInfo->bind_param("i", $id);
    $stmtInfo->execute();
    $resInfo = $stmtInfo->get_result();
    if ($resInfo && $resInfo->num_rows > 0) {
        $info = $resInfo->fetch_assoc();
        $enrolment_id = (int)$info['enrolment_id'];
        $bid = (int)$info['bid'];
    }
    $stmtInfo->close();

    if (!$enrolment_id) {
        throw new Exception("Enrolment not found for receipt.");
    }

    // === Step 3: Get course fee and discount from course-batch relation ===
    $course_fee = $discount = 0;
    $stmtFee = $conn->prepare("
        SELECT c.fees, c.discount
        FROM course c
        JOIN batch b ON b.cid = c.id
        WHERE b.id = ?
    ");
    $stmtFee->bind_param("i", $bid);
    $stmtFee->execute();
    $resFee = $stmtFee->get_result();
    if ($resFee && $resFee->num_rows > 0) {
        $rowFee = $resFee->fetch_assoc();
        $course_fee = (float)$rowFee['fees'];
        $discount = (float)$rowFee['discount'];
    }
    $stmtFee->close();

    $final_fee = $course_fee - $discount;

    // === Step 4: Get total paid (sum of all receipts with status Paid) ===
    $total_paid = 0;
    $stmtPaid = $conn->prepare("
        SELECT SUM(payment_amount) AS total_paid 
        FROM fees_receipts 
        WHERE enrolment_id = ? AND deleted_at IS NULL AND payment_status = 'Paid'
    ");
    $stmtPaid->bind_param("i", $enrolment_id);
    $stmtPaid->execute();
    $resPaid = $stmtPaid->get_result();
    if ($resPaid && $resPaid->num_rows > 0) {
        $total_paid = (float)($resPaid->fetch_assoc()['total_paid'] ?? 0);
    }
    $stmtPaid->close();

    // === Step 5: Determine new fee_status for enrolments ===
    $fee_status = 'Pending';
    if ($total_paid >= $final_fee) {
        $fee_status = 'Paid';
    } elseif ($total_paid > 0 && $total_paid < $final_fee) {
        $fee_status = 'Partial';
    }

    // === Step 6: Update enrolments table ===
    $stmtUpdate = $conn->prepare("UPDATE enrolments SET fee_status = ? WHERE enrolment_id = ?");
    $stmtUpdate->bind_param("si", $fee_status, $enrolment_id);
    $stmtUpdate->execute();
    $stmtUpdate->close();

    echo json_encode([
        'success' => true,
        'message' => 'Payment and enrolment status updated successfully.',
        'fee_status' => $fee_status,
        'total_paid' => $total_paid,
        'final_fee' => $final_fee
    ]);
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

$conn->close();
?>
