<?php
header('Content-Type: application/json');
include('../config.php');
session_start();

$response = ['success' => false, 'message' => 'Unknown error'];

// Ensure POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

// CSRF check
if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    exit(json_encode(["success" => false, "message" => "CSRF validation failed"]));
}

// Helper function
function sanitize($str) {
    return htmlspecialchars(trim($str), ENT_QUOTES, 'UTF-8');
}

// Get POST data
$action = sanitize($_POST['action'] ?? '');
$id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
$name = sanitize($_POST['name'] ?? '');
$email = sanitize($_POST['email'] ?? '');
$phone = sanitize($_POST['phone'] ?? '');
$role = sanitize($_POST['role'] ?? '');
$designation = sanitize($_POST['designation'] ?? '');

$password = trim($_POST['password'] ?? '');
$dob = sanitize($_POST['dob'] ?? '');
$bgroup = sanitize($_POST['blood_group'] ?? '');

// Validation
if (!in_array($action, ['delete', 'Activate', 'Deactivate']) && (empty($name) || empty($email) || empty($phone) || empty($role))) {
    echo json_encode(['success' => false, 'message' => 'Please fill all required fields.']);
    exit;
}
if (!in_array($action, ['delete', 'Activate', 'Deactivate']) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
    echo json_encode(['success' => false, 'message' => 'Invalid email format.']);
    exit;
}
if (!in_array($action, ['delete', 'Activate', 'Deactivate']) && !preg_match('/^[0-9]{10}$/', $phone)) {
    echo json_encode(['success' => false, 'message' => 'Enter valid 10 digit mobile number.']);
    exit;
}

// Email uniqueness
$stmt = $conn->prepare("SELECT user_id FROM users WHERE email = ?" . ($action === 'Edit' ? " AND user_id != ?" : ""));
if ($action === 'Edit') {
    $stmt->bind_param("si", $email, $id);
} else {
    $stmt->bind_param("s", $email);
}
$stmt->execute();
$stmt->store_result();
if ($stmt->num_rows > 0) {
    $stmt->close();
    echo json_encode(['success' => false, 'message' => 'Email already exists.']);
    exit;
}
$stmt->close();


// === ACTIVATE / DEACTIVATE USER ===
if (($action === 'Activate' || $action === 'Deactivate') && $id > 0) {
    $newStatus = ($action === 'Activate') ? 'Active' : 'Deactive';
    try {
        $stmt = $conn->prepare("UPDATE users SET status = ?, updated_at = NOW() WHERE user_id = ?");
        $stmt->bind_param("si", $newStatus, $id);
        if ($stmt->execute()) {
            $response = [
                'success' => true,
                'message' => "User {$newStatus}d successfully."
            ];
        } else {
            $response['message'] = "Failed to {$action} user.";
        }
        $stmt->close();
    } catch (Exception $e) {
        $response['message'] = 'Error: ' . $e->getMessage();
    }

    echo json_encode($response);
    $conn->close();
    exit;
}


// === ADD / EDIT / DELETE ===
try {
    if ($action === 'Add') {
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $conn->prepare("INSERT INTO users (name, email, phone, role, designation, password, dob, bgroup, created_at)
VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())
");
        $stmt->bind_param("ssssssss", $name, $email, $phone, $role, $designation, $hashedPassword, $dob, $bgroup);


        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'User added successfully', 'redirect' => 'users.php'];
        } else {
            $response['message'] = 'Failed to add user: ' . $stmt->error;
        }
        $stmt->close();

    } elseif ($action === 'Edit' && $id > 0) {
        $stmt = $conn->prepare("UPDATE users SET name=?, email=?, phone=?, role=?, designation=?, dob=?, bgroup=?, updated_at=NOW() WHERE user_id=?
");
        $stmt->bind_param("sssssssi", $name, $email, $phone, $role, $designation, $dob, $bgroup, $id);


        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'User updated successfully', 'redirect' => 'users.php'];
        } else {
            $response['message'] = 'Failed to update user: ' . $stmt->error;
        }
        $stmt->close();

    } elseif ($action === 'delete' && $id > 0) {
        $stmt = $conn->prepare("UPDATE users SET deleted_at = NOW() WHERE user_id = ?");
        $stmt->bind_param("i", $id);

        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'User deleted successfully.'];
        } else {
            $response['message'] = 'Failed to delete user.';
        }
        $stmt->close();

    } else {
        $response['message'] = 'Invalid action or missing user ID';
    }
} catch (Exception $e) {
    $response['message'] = 'Error: ' . $e->getMessage();
}

echo json_encode($response);
$conn->close();
exit;
?>
