<?php
include_once('./config.php');

if (session_status() === PHP_SESSION_NONE) session_start();

/* ------------------ CSRF CHECK ------------------ */
if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
    $_SESSION['flash'] = "Invalid request";
    header("Location: login.php");
    exit;
}

/* ------------------ INPUTS ------------------ */
$subscriber_id = intval($_POST['subscriber_id'] ?? 0);
$raw_token     = $_POST['raw_token'] ?? '';
$password      = $_POST['password'] ?? '';
$confirm       = $_POST['confirm_password'] ?? '';

if (!$subscriber_id || !$raw_token) {
    $_SESSION['flash'] = "Invalid reset link";
    header("Location: login.php");
    exit;
}

/* ------------------ PASSWORD CHECKS ------------------ */
if (strlen($password) < 8) {
    $_SESSION['flash'] = "Password must be at least 8 characters.";
    header("Location: reset.php?id=$subscriber_id&token=$raw_token");
    exit;
}
if ($password !== $confirm) {
    $_SESSION['flash'] = "Passwords do not match.";
    header("Location: reset.php?id=$subscriber_id&token=$raw_token");
    exit;
}

/* ------------------ FETCH TOKEN ------------------ */
$stmt = $conn->prepare("
    SELECT token, expires_at 
    FROM password_resets 
    WHERE subscriber_id = ? 
    LIMIT 1
");
$stmt->bind_param("i", $subscriber_id);
$stmt->execute();
$tokenData = $stmt->get_result()->fetch_assoc();

if (!$tokenData) {
    $_SESSION['flash'] = "Invalid or expired reset request.";
    header("Location: login.php");
    exit;
}

$hashed_token = $tokenData['token'];
$expires_at   = $tokenData['expires_at'];

/* ------------------ VERIFY TOKEN ------------------ */
if (!password_verify($raw_token, $hashed_token)) {
    $_SESSION['flash'] = "Invalid or expired reset token.";
    header("Location: login.php");
    exit;
}

if (strtotime($expires_at) < time()) {
    $_SESSION['flash'] = "Reset link expired.";
    header("Location: login.php");
    exit;
}

/* ------------------ UPDATE PASSWORD ------------------ */
$new_pass_hash = password_hash($password, PASSWORD_BCRYPT);

$update = $conn->prepare("UPDATE subscriber SET password = ?, updated_at = NOW() WHERE id = ?");
$update->bind_param("si", $new_pass_hash, $subscriber_id);

if (!$update->execute()) {
    $_SESSION['flash'] = "Could not update password. Try again.";
    header("Location: login.php");
    exit;
}

/* ------------------ DELETE USED TOKEN ------------------ */
$del = $conn->prepare("DELETE FROM password_resets WHERE subscriber_id = ?");
$del->bind_param("i", $subscriber_id);
$del->execute();

/* ------------------ SUCCESS ------------------ */
$_SESSION['flash'] = "Password updated successfully. Please login.";
header("Location: login.php");
exit;

