<?php
session_start();
header('Content-Type: application/json');
require_once '../config.php';
require_once 'auth.php';

class ChangePasswordHandler {
    private $conn;
    private $user_id;

    public function __construct($db, $user_id) {
        $this->conn = $db;
        $this->user_id = $user_id;
    }

    public function changePassword($old_password, $new_password, $confirm_password) {
        if ($new_password !== $confirm_password) {
            return $this->response(false, "New password and confirm password do not match.");
        }

        // Get current hashed password from DB
        $stmt = $this->conn->prepare("SELECT password FROM users WHERE user_id = ?");
        $stmt->bind_param("i", $this->user_id);
        $stmt->execute();
        $stmt->store_result();

        if ($stmt->num_rows !== 1) {
            return $this->response(false, "User not found.");
        }

        $stmt->bind_result($hashed_password_db);
        $stmt->fetch();

        // Verify old password
        if (!password_verify($old_password, $hashed_password_db)) {
            return $this->response(false, "Old password is incorrect.");
        }

        // Hash new password
        $new_hashed = password_hash($new_password, PASSWORD_DEFAULT);

        // Update password
        $update = $this->conn->prepare("UPDATE users SET password = ? WHERE user_id = ?");
        $update->bind_param("si", $new_hashed, $this->user_id);

        if ($update->execute()) {
            return $this->response(true, "Password changed successfully.");
        } else {
            return $this->response(false, "Failed to change password.");
        }
    }

    private function response($success, $message) {
        return json_encode([
            'success' => $success,
            'message' => $message
        ]);
    }
}

// Only accept JSON POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $data = json_decode(file_get_contents("php://input"), true);

    if (!isset($data['old_password'], $data['new_password'], $data['confirm_password'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid request! Make sure all inputs are filled.']);
        exit;
    }

    // Replace this with your actual user ID from session
    $user_id = $_SESSION['user_id']; // Ensure this is set in your login system

    $handler = new ChangePasswordHandler($conn, $user_id);
    echo $handler->changePassword(
        trim($data['old_password']),
        trim($data['new_password']),
        trim($data['confirm_password'])
    );
} else {
    echo json_encode(['success' => false, 'message' => 'Invalid request method.']);
}
?>
