<?php
header('Content-Type: application/json');
include('../config.php');
session_start();

// Ensure that the request method is POST
if ($_SERVER['REQUEST_METHOD'] != 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method.']);
    exit;
}


// CSRF check
if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    exit(json_encode(["status" => "error", "message" => "CSRF validation failed"]));
}

$response = ['success' => false, 'message' => 'Unknown error'];

// Simple role check (optional, can restrict deletes/edits if needed)
$role = $_SESSION['role'] ?? '';

// Get POST parameters
$action = $_POST['action'] ?? '';
$enquiry_id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
$name = $_POST['name'] ?? '';
$email = $_POST['email'] ?? '';
$phone = $_POST['phone'] ?? '';
$subject = $_POST['subject'] ?? '';
$message = $_POST['message'] ?? '';
$source = $_POST['source'] ?? 'Offline'; // Default 'Offline' if source is not provided
$reference = $_POST['reference'] ?? '';
$assigned_to = $_POST['assigned_to'] ?? '';

// Sanitize inputs to remove HTML, scripts, and unwanted characters
$name = htmlspecialchars(strip_tags($name), ENT_QUOTES, 'UTF-8');
$email = htmlspecialchars(strip_tags($email), ENT_QUOTES, 'UTF-8');
$phone = htmlspecialchars(strip_tags($phone), ENT_QUOTES, 'UTF-8');
$subject = htmlspecialchars(strip_tags($subject), ENT_QUOTES, 'UTF-8');
$message = htmlspecialchars(strip_tags($message), ENT_QUOTES, 'UTF-8');
$source = htmlspecialchars(strip_tags($source), ENT_QUOTES, 'UTF-8');
$reference = htmlspecialchars(strip_tags($reference), ENT_QUOTES, 'UTF-8');
$assigned_to = (int) htmlspecialchars(strip_tags($assigned_to), ENT_QUOTES, 'UTF-8'); // Sanitizing assigned_to as an integer

// Assign action
if ($action === 'assign') {
    // Ensure only Admin can assign enquiries
    if ($_SESSION['role'] !== 'Admin') {
        $response = ['success' => false, 'message' => 'Unauthorized action. Only Admins can assign enquiries.'];
    } else {
        try {
            // Update the 'assigned_to' field in the database
            $stmt = $conn->prepare("UPDATE enquiries SET assigned_to = ? WHERE enquiry_id = ?");
            $stmt->bind_param("ss", $assigned_to, $enquiry_id);

            if ($stmt->execute()) {
                $response = ['success' => true, 'message' => 'Enquiry assigned successfully'];
            } else {
                $response = ['success' => false, 'message' => 'Failed to assign enquiry. Please try again.'];
            }

            $stmt->close();
        } catch (Exception $e) {
            $response = ['success' => false, 'message' => 'Error assigning enquiry: ' . $e->getMessage()];
        }
    }
} elseif ($action === 'Add') {
    // Add new enquiry
    try {
        $stmt = $conn->prepare("INSERT INTO enquiries (name, email, phone, subject, message, source, reference) VALUES (?, ?, ?, ?, ?, ?, ?)");
        $stmt->bind_param("sssssss", $name, $email, $phone, $subject, $message, $source, $reference);

        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Enquiry added successfully', 'redirect' => 'enquiries.php'];
        } else {
            $response['message'] = 'Failed to add enquiry';
        }

        $stmt->close();
    } catch (Exception $e) {
        $response['message'] = 'Error adding enquiry: ' . $e->getMessage();
    }

} elseif ($action === 'Edit' && $enquiry_id > 0) {
    // Edit existing enquiry
    try {
        $stmt = $conn->prepare("UPDATE enquiries SET name=?, email=?, phone=?, subject=?, message=?, source=?, reference=?, updated_at=NOW() WHERE enquiry_id=?");
        $stmt->bind_param("sssssssi", $name, $email, $phone, $subject, $message, $source, $reference, $enquiry_id);

        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Enquiry updated successfully', 'redirect' => 'enquiries.php'];
        } else {
            $response['message'] = 'Failed to update enquiry';
        }

        $stmt->close();
    } catch (Exception $e) {
        $response['message'] = 'Error updating enquiry: ' . $e->getMessage();
    }
}elseif ($action === 'delete' && $enquiry_id > 0) {
    try {
        // Prepare and execute the soft delete query
        $stmt = $conn->prepare("UPDATE enquiries SET deleted_at = NOW() WHERE enquiry_id = ?");
        $stmt->bind_param("i", $enquiry_id);
        
        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Enquiry deleted successfully.'];
        } else {
            $response['message'] = 'Failed to delete enquiry.';
        }

        $stmt->close();
    } catch (Exception $e) {
        $response['message'] = 'Error: ' . $e->getMessage();
    }

} else {
    $response['message'] = 'Invalid request or action';
}


// Return JSON response
echo json_encode($response);
$conn->close();
exit;
?>


