<?php
header('Content-Type: application/json');
include('../config.php');
session_start();

// Ensure request is POST
if ($_SERVER['REQUEST_METHOD'] != 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method.']);
    exit;
}

// CSRF check
if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    echo json_encode(['success' => false, 'message' => 'CSRF validation failed']);
    exit;
}

$response = ['success' => false, 'message' => 'Unknown error'];

// Get POST parameters
$action = $_POST['action'] ?? '';
$id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
$bid = $_POST['bid'] ?? '';
$cid = $_POST['cid'] ?? '';
$mobile = $_POST['mobile'] ?? '';
$enrolment_date = $_POST['enrolment_date'] ?? '';

// Sanitize inputs
$bid = htmlspecialchars(strip_tags($bid), ENT_QUOTES, 'UTF-8');
$cid = htmlspecialchars(strip_tags($cid), ENT_QUOTES, 'UTF-8');
$mobile = htmlspecialchars(strip_tags($mobile), ENT_QUOTES, 'UTF-8');
$enrolment_date = htmlspecialchars(strip_tags($enrolment_date), ENT_QUOTES, 'UTF-8');

// Validate required fields (except for delete)
if ($action != 'delete' && (empty($bid) || empty($mobile) || empty($enrolment_date))) {
    echo json_encode(['success' => false, 'message' => 'Required fields cannot be empty.']);
    exit;
}

try {
    // DELETE ACTION (Soft Delete)
    if ($action === 'delete' && $id > 0) {
        $stmt = $conn->prepare("UPDATE enrolments SET deleted_at = NOW() WHERE enrolment_id = ?");
        $stmt->bind_param("i", $id);
        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Enrolement deleted successfully.'];
        } else {
            $response['message'] = 'Failed to delete enrolement.';
        }
        $stmt->close();
    }


// ADD ACTION
elseif ($action === 'Add') {
    // Check for duplicate enrolment
    $check = $conn->prepare("SELECT enrolment_id FROM enrolments 
                             WHERE cid = ? AND bid = ? AND mobile = ? AND deleted_at IS NULL");
    $check->bind_param("iis", $cid, $bid, $mobile);
    $check->execute();
    $check->store_result();

    if ($check->num_rows > 0) {
        // Duplicate found
        $response = [
            'success' => false,
            'message' => 'This student is already enrolled in the selected batch and course.'
        ];
    } else {
        // Proceed with insert
        $stmt = $conn->prepare("INSERT INTO enrolments (cid, bid, mobile, enrolment_date) VALUES (?, ?, ?, ?)");
        $stmt->bind_param("iiss", $cid, $bid, $mobile, $enrolment_date);
        if ($stmt->execute()) {
            $response = [
                'success' => true,
                'message' => 'Enrolment added successfully.',
                'redirect' => 'enrolements.php'
            ];
        } else {
            $response['message'] = 'Failed to add enrolment.';
        }
        $stmt->close();
    }
    $check->close();
}

// EDIT ACTION
elseif ($action === 'Edit' && $id > 0) {
    // Check for duplicate enrolment (excluding the current record)
    $check = $conn->prepare("SELECT enrolment_id FROM enrolments 
                             WHERE cid = ? AND bid = ? AND mobile = ? 
                             AND deleted_at IS NULL AND enrolment_id != ?");
    $check->bind_param("iisi", $cid, $bid, $mobile, $id);
    $check->execute();
    $check->store_result();

    if ($check->num_rows > 0) {
        // Duplicate found
        $response = [
            'success' => false,
            'message' => 'Another enrolment already exists with the same student, course, and batch.'
        ];
    } else {
        // Proceed with update
        $stmt = $conn->prepare("UPDATE enrolments 
                                SET cid=?, bid=?, mobile=?, enrolment_date=?, updated_at=NOW() 
                                WHERE enrolment_id=?");
        $stmt->bind_param("iissi", $cid, $bid, $mobile, $enrolment_date, $id);
        if ($stmt->execute()) {
            $response = [
                'success' => true,
                'message' => 'Enrolment updated successfully.',
                'redirect' => 'enrolements.php'
            ];
        } else {
            $response['message'] = 'Failed to update enrolment.';
        }
        $stmt->close();
    }
    $check->close();
}

    // INVALID ACTION
    else {
        $response['message'] = 'Invalid action or missing enrolement ID.';
    }

} catch (Exception $e) {
    $response['message'] = 'Error: ' . $e->getMessage();
}

// Return response
echo json_encode($response);
$conn->close();
exit;
?>



