<?php
session_start();
require "../config.php";

header('Content-Type: application/json');

// Ensure that the request method is POST
if ($_SERVER['REQUEST_METHOD'] != 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method.']);
    exit;
}

// CSRF check
if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    exit(json_encode(["status" => "error", "message" => "CSRF validation failed"]));
    exit;
}

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['status' => 'error', 'message' => 'User not logged in']);
    exit;
}

$userId = $_SESSION['user_id'];

// Validation & sanitization
$name = trim($_POST['name'] ?? '');
$email = trim($_POST['email'] ?? '');
$phone = trim($_POST['phone'] ?? '');

$errors = [];

if (!preg_match("/^[a-zA-Z0-9 ]+$/", $name)) {
    $errors[] = "Name can only contain letters, numbers, and spaces.";
}

if (!preg_match("/^[0-9]{10}$/", $phone)) {
    $errors[] = "Phone number must be exactly 10 digits.";
}

if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    $errors[] = "Invalid email format.";
}

if (!empty($errors)) {
    echo json_encode(['status' => 'error', 'message' => implode(' ', $errors)]);
    exit;
}

// Check email uniqueness
$stmt = $conn->prepare("SELECT user_id FROM users WHERE email = ? AND user_id != ?");
$stmt->bind_param("si", $email, $userId);
$stmt->execute();
$stmt->store_result();
if ($stmt->num_rows > 0) {
    $errors[] = "This email is already used by another user.";
}
$stmt->close();

if (!empty($errors)) {
    echo json_encode(['status' => false, 'message' => implode(' ', $errors)]);
    exit;
}

$uploadDir = './profile_img/';
$profileImage = null;

// Handle file upload if exists
if (!empty($_FILES['profile_image']['name'])) {
    $fileName = time() . '_' . basename($_FILES['profile_image']['name']);
    $targetFile = $uploadDir . $fileName;
    if (move_uploaded_file($_FILES['profile_image']['tmp_name'], $targetFile)) {
        $profileImage = 'profile_img/'.$fileName;
    } else {
        echo json_encode(['status' => 'error', 'message' => 'Failed to upload profile image']);
        exit;
    }
}

// Prepare update query
if ($profileImage) {
    $stmt = $conn->prepare("UPDATE users SET name=?, email=?, phone=?, profile_image=?, updated_at=NOW() WHERE user_id=?");
    $stmt->bind_param("ssssi", $name, $email, $phone, $profileImage, $userId);
} else {
    $stmt = $conn->prepare("UPDATE users SET name=?, email=?, phone=?, updated_at=NOW() WHERE user_id=?");
    $stmt->bind_param("sssi", $name, $email, $phone, $userId);
}

if ($stmt->execute()) {
    echo json_encode(['status' => 'success', 'message' => 'Profile updated successfully']);
} else {
    echo json_encode(['status' => 'error', 'message' => 'Failed to update profile']);
}
