<?php
header('Content-Type: application/json');
include('../config.php');
session_start();

// Ensure POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method.']);
    exit;
}

// CSRF check
if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    echo json_encode(['success' => false, 'message' => 'CSRF validation failed.']);
    exit;
}

$response = ['success' => false, 'message' => 'Unknown error.'];

// Read form data
$action = $_POST['action'] ?? '';
$id = isset($_POST['id']) ? (int)$_POST['id'] : 0;

$name = trim($_POST['title'] ?? '');
$description = trim($_POST['description'] ?? '');

// Validation
if (!in_array($action, ['delete', 'Activate', 'Deactivate']) && (empty($name) || empty($description))) {
    echo json_encode(['success' => false, 'message' => 'Service title and description are required.']);
    exit;
}


// Directory for uploads
$uploadDir = './assets/img/services/';
if (!is_dir($uploadDir)) {
    mkdir($uploadDir, 0777, true);
}

$imagePath = null;

// Handle file upload (for Add and Edit)
if (!empty($_FILES['image']['name'])) {
    $fileName = time() . '_' . preg_replace('/[^A-Za-z0-9._-]/', '_', $_FILES['image']['name']);
    $targetFile = $uploadDir . $fileName;
    $fileType = strtolower(pathinfo($targetFile, PATHINFO_EXTENSION));

    // Validate image type
    $allowedTypes = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
    if (!in_array($fileType, $allowedTypes)) {
        echo json_encode(['success' => false, 'message' => 'Invalid image format. Only JPG, PNG, GIF, WEBP allowed.']);
        exit;
    }

    // Move file
    if (move_uploaded_file($_FILES['image']['tmp_name'], $targetFile)) {
        $imagePath = 'assets/img/services/' . $fileName;

        // If editing and there was an old image, remove it
        if (!empty($_POST['old_image']) && file_exists('../' . $_POST['old_image'])) {
            @unlink('../' . $_POST['old_image']);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to upload image.']);
        exit;
    }
}


// === DELETE COURSE ===
if ($action === 'delete' && $id > 0) {
    try {
        $stmt = $conn->prepare("UPDATE services SET deleted_at = NOW() WHERE service_id = ?");
        $stmt->bind_param("i", $id);
        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Service deleted successfully.'];
        } else {
            $response['message'] = 'Failed to delete service.';
        }
        $stmt->close();
    } catch (Exception $e) {
        $response['message'] = 'Error: ' . $e->getMessage();
    }

    echo json_encode($response);
    $conn->close();
    exit;
}

// === ACTIVATE / DEACTIVATE COURSE ===
if (($action === 'Activate' || $action === 'Deactivate') && $id > 0) {
    $newStatus = ($action === 'Activate') ? 'Active' : 'Deactive';
    try {
        $stmt = $conn->prepare("UPDATE services SET status = ?, updated_at = NOW() WHERE service_id = ?");
        $stmt->bind_param("si", $newStatus, $id);
        if ($stmt->execute()) {
            $response = [
                'success' => true,
                'message' => "Service {$newStatus}d successfully."
            ];
        } else {
            $response['message'] = "Failed to {$action} service.";
        }
        $stmt->close();
    } catch (Exception $e) {
        $response['message'] = 'Error: ' . $e->getMessage();
    }

    echo json_encode($response);
    $conn->close();
    exit;
}


// === ADD NEW COURSE ===
if ($action === 'Add') {
    try {
        $stmt = $conn->prepare("INSERT INTO services (title, description, image) VALUES (?, ?, ?)");
        $stmt->bind_param("sss", $name, $description, $imagePath);
        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Service added successfully.', 'redirect' => 'services.php'];
        } else {
            $response['message'] = 'Failed to add service.';
        }
        $stmt->close();
    } catch (Exception $e) {
        $response['message'] = 'Error: ' . $e->getMessage();
    }
}

// === EDIT EXISTING COURSE ===
elseif ($action === 'Edit' && $id > 0) {
    try {
        if ($imagePath) {
            $stmt = $conn->prepare("UPDATE services SET title=?, description=?, image=?, updated_at=NOW() WHERE service_id=?");
            $stmt->bind_param("sssi", $name, $description, $imagePath, $id);
        } else {
            $stmt = $conn->prepare("UPDATE services SET title=?, description=?, updated_at=NOW() WHERE service_id=?");
            $stmt->bind_param("ssi", $name, $description, $id);
        }

        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Service updated successfully.', 'redirect' => 'services.php'];
        } else {
            $response['message'] = 'Failed to update service.';
        }

        $stmt->close();
    } catch (Exception $e) {
        $response['message'] = 'Error: ' . $e->getMessage();
    }
} else {
    $response['message'] = 'Invalid action or missing ID.';
}

echo json_encode($response);
$conn->close();
exit;
?>


