<?php
header('Content-Type: application/json');
include('../config.php');
session_start();

$response = ['success' => false, 'message' => 'Unknown error'];

// Ensure POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

// CSRF check
if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    exit(json_encode(["success" => false, "message" => "CSRF validation failed"]));
}

// Helper function
function sanitize($str) {
    return htmlspecialchars(trim($str), ENT_QUOTES, 'UTF-8');
}

$action = sanitize($_POST['action'] ?? '');
$id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
$name = sanitize($_POST['name'] ?? '');
$email = sanitize($_POST['email'] ?? '');
$phone = sanitize($_POST['mobile'] ?? '');
$reference = sanitize($_POST['reference'] ?? '');
$dob = sanitize($_POST['dob'] ?? '');
$password = '123456';

// Validation
if ($action != 'delete' && (empty($name) || empty($email) || empty($phone) || empty($dob))) {
    echo json_encode(['success' => false, 'message' => 'Please fill all required fields.']);
    exit;
}
if ($action != 'delete' && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
    echo json_encode(['success' => false, 'message' => 'Invalid email format.']);
    exit;
}
if ($action != 'delete' && !preg_match('/^[0-9]{10}$/', $phone)) {
    echo json_encode(['success' => false, 'message' => 'Enter valid 10 digit mobile number.']);
    exit;
}

// Check mobile uniqueness
$stmt = $conn->prepare("SELECT id FROM student WHERE mobile = ?" . ($action === 'Edit' ? " AND id != ?" : ""));
if ($action === 'Edit') {
    $stmt->bind_param("si", $phone, $id);
} else {
    $stmt->bind_param("s", $phone);
}
$stmt->execute();
$stmt->store_result();
if ($stmt->num_rows > 0) {
    $stmt->close();
    echo json_encode(['success' => false, 'message' => 'Mobile already exists for another student.']);
    exit;
}
$stmt->close();

try {
    // Handle ADD
    if ($action === 'Add') {

        // Generate registration number
        $prefix = $settings['registration_no_prefix'] ?? 'REG';
        $month = date('m');
        $year = date('y');
        $last4 = substr($phone, -4);
        $registration_no = $prefix . $month . $year . $last4;

        // Handle photo upload (required on Add)
        if (empty($_FILES['photo']['name'])) {
            echo json_encode(['success' => false, 'message' => 'Student photo is required.']);
            exit;
        }

        $photo_name = time() . '_' . basename($_FILES['photo']['name']);
        $target_dir = "./uploads/";
        if (!file_exists($target_dir)) mkdir($target_dir, 0777, true);
        $target_path = $target_dir . $photo_name;

        if (!move_uploaded_file($_FILES['photo']['tmp_name'], $target_path)) {
            echo json_encode(['success' => false, 'message' => 'Failed to upload photo.']);
            exit;
        }

        // Insert
        $stmt = $conn->prepare("INSERT INTO student (registration_no, name, email, mobile, dob, photo, reference, password) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->bind_param("ssssssss", $registration_no, $name, $email, $phone, $dob, $target_path, $reference, $password);

        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Student added successfully', 'redirect' => 'students.php'];
        } else {
            $response['message'] = 'Failed to add student: ' . $stmt->error;
        }
        $stmt->close();
    }

    // Handle EDIT
    elseif ($action === 'Edit' && $id > 0) {

        // Handle optional photo update
        if (!empty($_FILES['photo']['name'])) {
            $photo_name = time() . '_' . basename($_FILES['photo']['name']);
            $target_dir = "../uploads/students/";
            if (!file_exists($target_dir)) mkdir($target_dir, 0777, true);
            $target_path = $target_dir . $photo_name;
            move_uploaded_file($_FILES['photo']['tmp_name'], $target_path);
        } else {
            $target_path = $_POST['old_photo'] ?? '';
        }

        $stmt = $conn->prepare("UPDATE student SET name=?, email=?, mobile=?, dob=?, photo=?, reference=? WHERE id=?");
        $stmt->bind_param("ssssssi", $name, $email, $phone, $dob, $target_path, $reference, $id);

        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Student updated successfully', 'redirect' => 'students.php'];
        } else {
            $response['message'] = 'Failed to update student: ' . $stmt->error;
        }
        $stmt->close();
    }

    // Handle DELETE
    elseif ($action === 'delete' && $id > 0) {
        $stmt = $conn->prepare("DELETE FROM student WHERE id = ?");
        $stmt->bind_param("i", $id);

        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'Student deleted successfully.'];
        } else {
            $response['message'] = 'Failed to delete student.';
        }

        $stmt->close();
    }

    else {
        $response['message'] = 'Invalid action or missing student ID';
    }

} catch (Exception $e) {
    $response['message'] = 'Error: ' . $e->getMessage();
}

echo json_encode($response);
$conn->close();
exit;
?>





